const express = require('express');
const router = express.Router();
const db = require('../db');
const bcrypt = require('bcryptjs');

// Middleware to check login
const requireLogin = (req, res, next) => {
    if (!req.session.loggedin) {
        return res.redirect('/login');
    }
    next();
};

// Middleware to check Admin
const requireAdmin = (req, res, next) => {
    if (req.session.role !== 'admin') {
        return res.status(403).send('Access Denied');
    }
    next();
};

// Dashboard
router.get('/', requireLogin, (req, res) => {
    res.render('dashboard', { 
        user: req.session.username,
        role: req.session.role 
    });
});

// --- Raw Materials ---
router.get('/raw-materials', requireLogin, async (req, res) => {
    const [rows] = await db.execute('SELECT * FROM raw_materials');
    res.render('raw-materials', { 
        items: rows, 
        role: req.session.role,
        user: req.session.username
    });
});

router.post('/raw-materials/add', requireLogin, requireAdmin, async (req, res) => {
    const { code, description } = req.body;
    await db.execute('INSERT INTO raw_materials (code, description) VALUES (?, ?)', [code, description]);
    res.redirect('/raw-materials');
});

router.post('/raw-materials/edit/:id', requireLogin, requireAdmin, async (req, res) => {
    const { code, description } = req.body;
    await db.execute('UPDATE raw_materials SET code=?, description=? WHERE id=?', [code, description, req.params.id]);
    res.redirect('/raw-materials');
});

// --- Resin PHRs ---
router.get('/resin-phrs', requireLogin, async (req, res) => {
    const [rows] = await db.execute('SELECT * FROM resin_phrs');
    res.render('resin-phrs', { 
        items: rows, 
        role: req.session.role,
        user: req.session.username
    });
});

router.post('/resin-phrs/add', requireLogin, requireAdmin, async (req, res) => {
    const { resin, phr, set_time } = req.body;
    await db.execute('INSERT INTO resin_phrs (resin, phr, set_time) VALUES (?, ?, ?)', [resin, phr, set_time]);
    res.redirect('/resin-phrs');
});

router.post('/resin-phrs/edit/:id', requireLogin, requireAdmin, async (req, res) => {
    const { resin, phr, set_time } = req.body;
    await db.execute('UPDATE resin_phrs SET resin=?, phr=?, set_time=? WHERE id=?', [resin, phr, set_time, req.params.id]);
    res.redirect('/resin-phrs');
});

// --- User Management (Admin Only) ---
router.get('/users', requireLogin, requireAdmin, async (req, res) => {
    const [rows] = await db.execute('SELECT id, username, role FROM users');
    res.render('users', { users: rows });
});

router.post('/users/add', requireLogin, requireAdmin, async (req, res) => {
    const { username, password, role } = req.body;
    const hash = await bcrypt.hash(password, 10);
    try {
        await db.execute('INSERT INTO users (username, password, role) VALUES (?, ?, ?)', [username, hash, role]);
        res.redirect('/users');
    } catch (e) {
        res.send('Error adding user (Username likely exists)');
    }
});

module.exports = router;